/*-
 * #%L
 * athena-cloudwatch
 * %%
 * Copyright (C) 2019 Amazon Web Services
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */
package com.amazonaws.athena.connectors.cloudwatch;

import com.amazonaws.athena.connector.lambda.QueryStatusChecker;
import com.amazonaws.athena.connector.lambda.ThrottlingInvoker;
import com.amazonaws.athena.connector.lambda.data.Block;
import com.amazonaws.athena.connector.lambda.data.BlockSpiller;
import com.amazonaws.athena.connector.lambda.domain.Split;
import com.amazonaws.athena.connector.lambda.domain.TableName;
import com.amazonaws.athena.connector.lambda.domain.predicate.Constraints;
import com.amazonaws.athena.connector.lambda.domain.predicate.Range;
import com.amazonaws.athena.connector.lambda.domain.predicate.SortedRangeSet;
import com.amazonaws.athena.connector.lambda.domain.predicate.ValueSet;
import com.amazonaws.athena.connector.lambda.handlers.RecordHandler;
import com.amazonaws.athena.connector.lambda.records.ReadRecordsRequest;
import com.amazonaws.athena.connectors.cloudwatch.qpt.CloudwatchQueryPassthrough;
import org.apache.arrow.util.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.services.athena.AthenaClient;
import software.amazon.awssdk.services.cloudwatchlogs.CloudWatchLogsClient;
import software.amazon.awssdk.services.cloudwatchlogs.model.GetLogEventsRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.GetLogEventsResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.GetQueryResultsResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.OutputLogEvent;
import software.amazon.awssdk.services.cloudwatchlogs.model.ResultField;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.secretsmanager.SecretsManagerClient;

import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicLong;

import static com.amazonaws.athena.connectors.cloudwatch.CloudwatchExceptionFilter.EXCEPTION_FILTER;
import static com.amazonaws.athena.connectors.cloudwatch.CloudwatchMetadataHandler.LOG_GROUP_FIELD;
import static com.amazonaws.athena.connectors.cloudwatch.CloudwatchMetadataHandler.LOG_MSG_FIELD;
import static com.amazonaws.athena.connectors.cloudwatch.CloudwatchMetadataHandler.LOG_STREAM_FIELD;
import static com.amazonaws.athena.connectors.cloudwatch.CloudwatchMetadataHandler.LOG_TIME_FIELD;
import static com.amazonaws.athena.connectors.cloudwatch.CloudwatchUtils.getResult;

/**
 * Handles data read record requests for the Athena Cloudwatch Connector.
 * <p>
 * For more detail, please see the module's README.md, some notable characteristics of this class include:
 * <p>
 * 1. Reads and maps Cloudwatch Logs data for a specific LogStream (split)
 * 2. Attempts to push down time range predicates into Cloudwatch.
 */
public class CloudwatchRecordHandler
        extends RecordHandler
{
    private static final Logger logger = LoggerFactory.getLogger(CloudwatchRecordHandler.class);
    //Used to tag log lines generated by this connector for diagnostic purposes when interacting with Athena.
    private static final String SOURCE_TYPE = "cloudwatch";
    //Used to handle Throttling events and apply AIMD congestion control
    private final ThrottlingInvoker invoker;
    private final AtomicLong count = new AtomicLong(0);
    private final CloudWatchLogsClient awsLogs;
    private final CloudwatchQueryPassthrough queryPassthrough = new CloudwatchQueryPassthrough();

    public CloudwatchRecordHandler(java.util.Map<String, String> configOptions)
    {
        this(
                S3Client.create(),
                SecretsManagerClient.create(),
                AthenaClient.create(),
                CloudWatchLogsClient.create(),
                configOptions);
    }

    @VisibleForTesting
    protected CloudwatchRecordHandler(S3Client amazonS3, SecretsManagerClient secretsManager, AthenaClient athena, CloudWatchLogsClient awsLogs, java.util.Map<String, String> configOptions)
    {
        super(amazonS3, secretsManager, athena, SOURCE_TYPE, configOptions);
        this.awsLogs = awsLogs;
        this.invoker = ThrottlingInvoker.newDefaultBuilder(EXCEPTION_FILTER, configOptions).build();
    }

    /**
     * Scans Cloudwatch Logs using the LogStream and optional Time stamp filters.
     *
     * @see RecordHandler
     */
    @Override
    protected void readWithConstraint(BlockSpiller spiller, ReadRecordsRequest recordsRequest, QueryStatusChecker queryStatusChecker)
            throws TimeoutException, InterruptedException
    {
        if (recordsRequest.getConstraints().isQueryPassThrough()) {
            getQueryPassthreoughResults(spiller, recordsRequest);
        }
        else {
            String continuationToken = null;
            TableName tableName = recordsRequest.getTableName();
            Split split = recordsRequest.getSplit();
            invoker.setBlockSpiller(spiller);
            do {
                final String actualContinuationToken = continuationToken;
                GetLogEventsResponse logEventsResponse = invoker.invoke(() -> awsLogs.getLogEvents(
                        pushDownConstraints(recordsRequest.getConstraints(),
                                GetLogEventsRequest.builder()
                                        .logGroupName(split.getProperty(LOG_GROUP_FIELD))
                                        //We use the property instead of the table name because of the special all_streams table
                                        .logStreamName(split.getProperty(LOG_STREAM_FIELD))
                                        .nextToken(actualContinuationToken)
                                        // must be set to use nextToken correctly
                                        .startFromHead(true)
                                        .build()
                        )));

                if (continuationToken == null || !continuationToken.equals(logEventsResponse.nextForwardToken())) {
                    continuationToken = logEventsResponse.nextForwardToken();
                }
                else {
                    continuationToken = null;
                }

                for (OutputLogEvent ole : logEventsResponse.events()) {
                    spiller.writeRows((Block block, int rowNum) -> {
                        boolean matched = true;
                        matched &= block.offerValue(LOG_STREAM_FIELD, rowNum, split.getProperty(LOG_STREAM_FIELD));
                        matched &= block.offerValue(LOG_TIME_FIELD, rowNum, ole.timestamp());
                        matched &= block.offerValue(LOG_MSG_FIELD, rowNum, ole.message());
                        return matched ? 1 : 0;
                    });
                }

                logger.info("readWithConstraint: LogGroup[{}] LogStream[{}] Continuation[{}] rows[{}]",
                        tableName.getSchemaName(), tableName.getTableName(), continuationToken,
                        logEventsResponse.events().size());
            }
            while (continuationToken != null && queryStatusChecker.isQueryRunning());
        }
    }

    private void getQueryPassthreoughResults(BlockSpiller spiller, ReadRecordsRequest recordsRequest) throws TimeoutException, InterruptedException
    {
        Map<String, String> qptArguments = recordsRequest.getConstraints().getQueryPassthroughArguments();
        queryPassthrough.verify(qptArguments);
        GetQueryResultsResponse getQueryResultsResponse = getResult(invoker, awsLogs, qptArguments, Integer.parseInt(qptArguments.get(CloudwatchQueryPassthrough.LIMIT)));

        for (List<ResultField> resultList : getQueryResultsResponse.results()) {
            spiller.writeRows((Block block, int rowNum) -> {
                for (ResultField resultField : resultList) {
                    boolean matched = true;
                    matched &= block.offerValue(resultField.field(), rowNum, resultField.value());
                    if (!matched) {
                        return 0;
                    }
                }
                return 1;
            });
        }
    }

    /**
     * Attempts to push down predicates into Cloudwatch Logs by decorating the Cloudwatch Logs request.
     *
     * @param constraints The constraints for the read as provided by Athena based on the customer's query.
     * @param request     The Cloudwatch Logs request to inject predicates to.
     * @return The decorated Cloudwatch Logs request.
     * @note This impl currently only pushing down SortedRangeSet filters (>=, =<, between) on the log time column.
     */
    private GetLogEventsRequest pushDownConstraints(Constraints constraints, GetLogEventsRequest request)
    {
        GetLogEventsRequest.Builder requestBuilder = request.toBuilder();
        ValueSet timeConstraint = constraints.getSummary().get(LOG_TIME_FIELD);
        if (timeConstraint instanceof SortedRangeSet && !timeConstraint.isNullAllowed()) {
            //SortedRangeSet is how >, <, between is represented which are easiest and most common when
            //searching logs so we attempt to push that down here as an optimization. SQL can represent complex
            //overlapping ranges which Cloudwatch can not support so this is not a replacement for applying
            //constraints using the ConstraintEvaluator.

            Range basicPredicate = ((SortedRangeSet) timeConstraint).getSpan();

            if (!basicPredicate.getLow().isNullValue()) {
                Long lowerBound = (Long) basicPredicate.getLow().getValue();
                requestBuilder.startTime(lowerBound);
            }

            if (!basicPredicate.getHigh().isNullValue()) {
                Long upperBound = (Long) basicPredicate.getHigh().getValue();
                requestBuilder.endTime(upperBound);
            }
        }

        return requestBuilder.build();
    }
}
