package com.datadog.android.log.model

import com.datadog.android.core.`internal`.utils.JsonSerializer
import com.google.gson.JsonArray
import com.google.gson.JsonElement
import com.google.gson.JsonObject
import com.google.gson.JsonParseException
import com.google.gson.JsonParser
import com.google.gson.JsonPrimitive
import java.lang.IllegalStateException
import java.lang.NullPointerException
import java.lang.NumberFormatException
import kotlin.Any
import kotlin.Array
import kotlin.Boolean
import kotlin.String
import kotlin.collections.ArrayList
import kotlin.collections.List
import kotlin.collections.MutableMap
import kotlin.jvm.JvmStatic
import kotlin.jvm.Throws

/**
 * Structure holding information about a Log
 * @param status The severity of this log
 * @param service The service name
 * @param message The log message
 * @param date The date when the log is fired as an ISO-8601 String
 * @param logger Information about the logger that produced this log.
 * @param dd Datadog internal information
 * @param usr User properties
 * @param account Account properties
 * @param network The network information in the moment the log was created
 * @param error The additional error information in case this log is marked as an error
 * @param buildId Generated unique ID of the application build. Unlike version or build_version this
 * field is not meant to be coming from the user, but rather generated by the tooling for each build.
 * @param ddtags The list of tags joined into a String and divided by ',' 
 */
public data class LogEvent(
    public var status: Status,
    public val service: String,
    public var message: String,
    public val date: String,
    public val logger: Logger,
    public val dd: Dd,
    public val usr: Usr? = null,
    public val account: Account? = null,
    public val network: Network? = null,
    public val error: Error? = null,
    public val buildId: String? = null,
    public var ddtags: String,
    public val additionalProperties: MutableMap<String, Any?> = mutableMapOf(),
) {
    public fun toJson(): JsonElement {
        val json = JsonObject()
        json.add("status", status.toJson())
        json.addProperty("service", service)
        json.addProperty("message", message)
        json.addProperty("date", date)
        json.add("logger", logger.toJson())
        json.add("_dd", dd.toJson())
        usr?.let { usrNonNull ->
            json.add("usr", usrNonNull.toJson())
        }
        account?.let { accountNonNull ->
            json.add("account", accountNonNull.toJson())
        }
        network?.let { networkNonNull ->
            json.add("network", networkNonNull.toJson())
        }
        error?.let { errorNonNull ->
            json.add("error", errorNonNull.toJson())
        }
        buildId?.let { buildIdNonNull ->
            json.addProperty("build_id", buildIdNonNull)
        }
        json.addProperty("ddtags", ddtags)
        additionalProperties.forEach { (k, v) ->
            if (k !in RESERVED_PROPERTIES) {
                json.add(k, JsonSerializer.toJsonElement(v))
            }
        }
        return json
    }

    public companion object {
        internal val RESERVED_PROPERTIES: Array<String> = arrayOf("status", "service", "message",
                "date", "logger", "_dd", "usr", "account", "network", "error", "build_id", "ddtags")

        @JvmStatic
        @Throws(JsonParseException::class)
        public fun fromJson(jsonString: String): LogEvent {
            try {
                val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                return fromJsonObject(jsonObject)
            } catch (e: IllegalStateException) {
                throw JsonParseException(
                    "Unable to parse json into type LogEvent",
                    e
                )
            }
        }

        @JvmStatic
        @Throws(JsonParseException::class)
        public fun fromJsonObject(jsonObject: JsonObject): LogEvent {
            try {
                val status = Status.fromJson(jsonObject.get("status").asString)
                val service = jsonObject.get("service").asString
                val message = jsonObject.get("message").asString
                val date = jsonObject.get("date").asString
                val logger = jsonObject.get("logger").asJsonObject.let {
                    Logger.fromJsonObject(it)
                }
                val dd = jsonObject.get("_dd").asJsonObject.let {
                    Dd.fromJsonObject(it)
                }
                val usr = jsonObject.get("usr")?.asJsonObject?.let {
                    Usr.fromJsonObject(it)
                }
                val account = jsonObject.get("account")?.asJsonObject?.let {
                    Account.fromJsonObject(it)
                }
                val network = jsonObject.get("network")?.asJsonObject?.let {
                    Network.fromJsonObject(it)
                }
                val error = jsonObject.get("error")?.asJsonObject?.let {
                    Error.fromJsonObject(it)
                }
                val buildId = jsonObject.get("build_id")?.asString
                val ddtags = jsonObject.get("ddtags").asString
                val additionalProperties = mutableMapOf<String, Any?>()
                for (entry in jsonObject.entrySet()) {
                    if (entry.key !in RESERVED_PROPERTIES) {
                        additionalProperties[entry.key] = entry.value
                    }
                }
                return LogEvent(status, service, message, date, logger, dd, usr, account, network,
                        error, buildId, ddtags, additionalProperties)
            } catch (e: IllegalStateException) {
                throw JsonParseException(
                    "Unable to parse json into type LogEvent",
                    e
                )
            } catch (e: NumberFormatException) {
                throw JsonParseException(
                    "Unable to parse json into type LogEvent",
                    e
                )
            } catch (e: NullPointerException) {
                throw JsonParseException(
                    "Unable to parse json into type LogEvent",
                    e
                )
            }
        }
    }

    /**
     * Information about the logger that produced this log.
     * @param name The name of the logger
     * @param threadName The thread name on which the log event was created
     * @param version The SDK version name
     */
    public data class Logger(
        public var name: String,
        public val threadName: String? = null,
        public val version: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("name", name)
            threadName?.let { threadNameNonNull ->
                json.addProperty("thread_name", threadNameNonNull)
            }
            json.addProperty("version", version)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Logger {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Logger",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Logger {
                try {
                    val name = jsonObject.get("name").asString
                    val threadName = jsonObject.get("thread_name")?.asString
                    val version = jsonObject.get("version").asString
                    return Logger(name, threadName, version)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Logger",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Logger",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Logger",
                        e
                    )
                }
            }
        }
    }

    /**
     * Datadog internal information
     * @param device Information about the device that produced this log.
     */
    public data class Dd(
        public val device: Device,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.add("device", device.toJson())
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Dd {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Dd {
                try {
                    val device = jsonObject.get("device").asJsonObject.let {
                        Device.fromJsonObject(it)
                    }
                    return Dd(device)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Dd",
                        e
                    )
                }
            }
        }
    }

    /**
     * User properties
     * @param id Identifier of the user
     * @param name Name of the user
     * @param email Email of the user
     */
    public data class Usr(
        public val id: String? = null,
        public val name: String? = null,
        public val email: String? = null,
        public val additionalProperties: MutableMap<String, Any?> = mutableMapOf(),
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            id?.let { idNonNull ->
                json.addProperty("id", idNonNull)
            }
            name?.let { nameNonNull ->
                json.addProperty("name", nameNonNull)
            }
            email?.let { emailNonNull ->
                json.addProperty("email", emailNonNull)
            }
            additionalProperties.forEach { (k, v) ->
                if (k !in RESERVED_PROPERTIES) {
                    json.add(k, JsonSerializer.toJsonElement(v))
                }
            }
            return json
        }

        public companion object {
            internal val RESERVED_PROPERTIES: Array<String> = arrayOf("id", "name", "email")

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Usr {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Usr",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Usr {
                try {
                    val id = jsonObject.get("id")?.asString
                    val name = jsonObject.get("name")?.asString
                    val email = jsonObject.get("email")?.asString
                    val additionalProperties = mutableMapOf<String, Any?>()
                    for (entry in jsonObject.entrySet()) {
                        if (entry.key !in RESERVED_PROPERTIES) {
                            additionalProperties[entry.key] = entry.value
                        }
                    }
                    return Usr(id, name, email, additionalProperties)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Usr",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Usr",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Usr",
                        e
                    )
                }
            }
        }
    }

    /**
     * Account properties
     * @param id Identifier of the account
     * @param name Name of the account
     */
    public data class Account(
        public val id: String? = null,
        public val name: String? = null,
        public val additionalProperties: MutableMap<String, Any?> = mutableMapOf(),
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            id?.let { idNonNull ->
                json.addProperty("id", idNonNull)
            }
            name?.let { nameNonNull ->
                json.addProperty("name", nameNonNull)
            }
            additionalProperties.forEach { (k, v) ->
                if (k !in RESERVED_PROPERTIES) {
                    json.add(k, JsonSerializer.toJsonElement(v))
                }
            }
            return json
        }

        public companion object {
            internal val RESERVED_PROPERTIES: Array<String> = arrayOf("id", "name")

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Account {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Account",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Account {
                try {
                    val id = jsonObject.get("id")?.asString
                    val name = jsonObject.get("name")?.asString
                    val additionalProperties = mutableMapOf<String, Any?>()
                    for (entry in jsonObject.entrySet()) {
                        if (entry.key !in RESERVED_PROPERTIES) {
                            additionalProperties[entry.key] = entry.value
                        }
                    }
                    return Account(id, name, additionalProperties)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Account",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Account",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Account",
                        e
                    )
                }
            }
        }
    }

    /**
     * The network information in the moment the log was created
     */
    public data class Network(
        public val client: Client,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.add("client", client.toJson())
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Network {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Network",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Network {
                try {
                    val client = jsonObject.get("client").asJsonObject.let {
                        Client.fromJsonObject(it)
                    }
                    return Network(client)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Network",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Network",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Network",
                        e
                    )
                }
            }
        }
    }

    /**
     * The additional error information in case this log is marked as an error
     * @param kind The kind of this error. It is resolved from the throwable class name
     * @param message The error message
     * @param stack The error stack trace
     * @param sourceType The source_type of the error (e.g. 'android', 'flutter', 'react-native')
     * @param fingerprint A custom fingerprint for this error
     * @param threads Description of each thread in the process when error happened.
     */
    public data class Error(
        public var kind: String? = null,
        public var message: String? = null,
        public var stack: String? = null,
        public var sourceType: String? = null,
        public var fingerprint: String? = null,
        public val threads: List<Thread>? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            kind?.let { kindNonNull ->
                json.addProperty("kind", kindNonNull)
            }
            message?.let { messageNonNull ->
                json.addProperty("message", messageNonNull)
            }
            stack?.let { stackNonNull ->
                json.addProperty("stack", stackNonNull)
            }
            sourceType?.let { sourceTypeNonNull ->
                json.addProperty("source_type", sourceTypeNonNull)
            }
            fingerprint?.let { fingerprintNonNull ->
                json.addProperty("fingerprint", fingerprintNonNull)
            }
            threads?.let { threadsNonNull ->
                val threadsArray = JsonArray(threadsNonNull.size)
                threadsNonNull.forEach { threadsArray.add(it.toJson()) }
                json.add("threads", threadsArray)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Error {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Error",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Error {
                try {
                    val kind = jsonObject.get("kind")?.asString
                    val message = jsonObject.get("message")?.asString
                    val stack = jsonObject.get("stack")?.asString
                    val sourceType = jsonObject.get("source_type")?.asString
                    val fingerprint = jsonObject.get("fingerprint")?.asString
                    val threads = jsonObject.get("threads")?.asJsonArray?.let { jsonArray ->
                        val collection = ArrayList<Thread>(jsonArray.size())
                        jsonArray.forEach {
                            collection.add(Thread.fromJsonObject(it.asJsonObject))
                        }
                        collection
                    }
                    return Error(kind, message, stack, sourceType, fingerprint, threads)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Error",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Error",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Error",
                        e
                    )
                }
            }
        }
    }

    /**
     * Information about the device that produced this log.
     * @param architecture The CPU architecture of the device
     */
    public data class Device(
        public val architecture: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("architecture", architecture)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Device {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Device {
                try {
                    val architecture = jsonObject.get("architecture").asString
                    return Device(architecture)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Device",
                        e
                    )
                }
            }
        }
    }

    /**
     * @param connectivity The active network
     */
    public data class Client(
        public val simCarrier: SimCarrier? = null,
        public val signalStrength: String? = null,
        public val downlinkKbps: String? = null,
        public val uplinkKbps: String? = null,
        public val connectivity: String,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            simCarrier?.let { simCarrierNonNull ->
                json.add("sim_carrier", simCarrierNonNull.toJson())
            }
            signalStrength?.let { signalStrengthNonNull ->
                json.addProperty("signal_strength", signalStrengthNonNull)
            }
            downlinkKbps?.let { downlinkKbpsNonNull ->
                json.addProperty("downlink_kbps", downlinkKbpsNonNull)
            }
            uplinkKbps?.let { uplinkKbpsNonNull ->
                json.addProperty("uplink_kbps", uplinkKbpsNonNull)
            }
            json.addProperty("connectivity", connectivity)
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Client {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Client",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Client {
                try {
                    val simCarrier = jsonObject.get("sim_carrier")?.asJsonObject?.let {
                        SimCarrier.fromJsonObject(it)
                    }
                    val signalStrength = jsonObject.get("signal_strength")?.asString
                    val downlinkKbps = jsonObject.get("downlink_kbps")?.asString
                    val uplinkKbps = jsonObject.get("uplink_kbps")?.asString
                    val connectivity = jsonObject.get("connectivity").asString
                    return Client(simCarrier, signalStrength, downlinkKbps, uplinkKbps,
                            connectivity)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Client",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Client",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Client",
                        e
                    )
                }
            }
        }
    }

    /**
     * Description of the thread in the process when error happened.
     * @param name Name of the thread (e.g. 'Thread 0').
     * @param crashed Tells if the thread crashed.
     * @param stack Unsymbolicated stack trace of the given thread.
     * @param state Platform-specific state of the thread when its state was captured (CPU registers
     * dump for iOS, thread state enum for Android, etc.).
     */
    public data class Thread(
        public val name: String,
        public val crashed: Boolean,
        public val stack: String,
        public val state: String? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            json.addProperty("name", name)
            json.addProperty("crashed", crashed)
            json.addProperty("stack", stack)
            state?.let { stateNonNull ->
                json.addProperty("state", stateNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): Thread {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Thread",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): Thread {
                try {
                    val name = jsonObject.get("name").asString
                    val crashed = jsonObject.get("crashed").asBoolean
                    val stack = jsonObject.get("stack").asString
                    val state = jsonObject.get("state")?.asString
                    return Thread(name, crashed, stack, state)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type Thread",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type Thread",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type Thread",
                        e
                    )
                }
            }
        }
    }

    public data class SimCarrier(
        public val id: String? = null,
        public val name: String? = null,
    ) {
        public fun toJson(): JsonElement {
            val json = JsonObject()
            id?.let { idNonNull ->
                json.addProperty("id", idNonNull)
            }
            name?.let { nameNonNull ->
                json.addProperty("name", nameNonNull)
            }
            return json
        }

        public companion object {
            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJson(jsonString: String): SimCarrier {
                try {
                    val jsonObject = JsonParser.parseString(jsonString).asJsonObject
                    return fromJsonObject(jsonObject)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type SimCarrier",
                        e
                    )
                }
            }

            @JvmStatic
            @Throws(JsonParseException::class)
            public fun fromJsonObject(jsonObject: JsonObject): SimCarrier {
                try {
                    val id = jsonObject.get("id")?.asString
                    val name = jsonObject.get("name")?.asString
                    return SimCarrier(id, name)
                } catch (e: IllegalStateException) {
                    throw JsonParseException(
                        "Unable to parse json into type SimCarrier",
                        e
                    )
                } catch (e: NumberFormatException) {
                    throw JsonParseException(
                        "Unable to parse json into type SimCarrier",
                        e
                    )
                } catch (e: NullPointerException) {
                    throw JsonParseException(
                        "Unable to parse json into type SimCarrier",
                        e
                    )
                }
            }
        }
    }

    /**
     * The severity of this log
     */
    public enum class Status(
        private val jsonValue: String,
    ) {
        CRITICAL("critical"),
        ERROR("error"),
        WARN("warn"),
        INFO("info"),
        DEBUG("debug"),
        TRACE("trace"),
        EMERGENCY("emergency"),
        ;

        public fun toJson(): JsonElement = JsonPrimitive(jsonValue)

        public companion object {
            @JvmStatic
            public fun fromJson(jsonString: String): Status = values().first {
                it.jsonValue == jsonString
            }
        }
    }
}
